<?php
/**
 * Part of Windwalker project. 
 *
 * @copyright  Copyright (C) 2016 LYRASOFT. All rights reserved.
 * @license    GNU General Public License version 2 or later.
 */

namespace Windwalker;

use Windwalker\DI\Container;
use Windwalker\Filesystem\Path\PathCollection;
use Windwalker\Joomla\DataMapper\DataMapperProvider;
use Windwalker\Provider\SystemProvider;

/**
 * Windwalker main application.
 *
 * @since 2.0
 */
class Windwalker
{
	/**
	 * Init windalkwer.
	 *
	 * @param bool $cli
	 *
	 * @throws \Exception
	 *
	 * @return void
	 */
	public function init($cli = false)
	{
		$version = new \JVersion;

		if (!$version->isCompatible('3.2'))
		{
			throw new \Exception('Windwalker need Joomla! 3.2 or higher.');
		}

		// Import Windwalker autoload.
		$this->autoload();

		include_once __DIR__ . '/PHP/methods.php';

		// Do not use realpath() since we use symlink to develop RAD now.
		define('WINDWALKER',        JPATH_LIBRARIES . DIRECTORY_SEPARATOR . 'windwalker');
		define('WINDWALKER_SOURCE', WINDWALKER . DIRECTORY_SEPARATOR . 'src');
		define('WINDWALKER_TEST',   WINDWALKER . DIRECTORY_SEPARATOR . 'test');
		define('WINDWALKER_BUNDLE', dirname(WINDWALKER) . DIRECTORY_SEPARATOR . 'windwalker-bundles');

		// Register global provider
		$container = Container::getInstance();

		$container->registerServiceProvider(new SystemProvider($cli))
			->registerServiceProvider(new DataMapperProvider);

		// Register bundles
		$this->registerBundles($container);

		// Load language
		$lang = $container->get('language');
		$lang->load('lib_windwalker', JPATH_BASE, null, false, false)
		|| $lang->load('lib_windwalker', WINDWALKER, null, false, false)
		|| $lang->load('lib_windwalker', JPATH_BASE, $lang->getDefault(), false, false)
		|| $lang->load('lib_windwalker', WINDWALKER, $lang->getDefault(), false, false);
	}

	/**
	 * Register Bundles
	 *
	 * @param Container $container DI container.
	 *
	 * @return  void
	 */
	protected function registerBundles(Container $container)
	{
		if (! is_dir(WINDWALKER_BUNDLE))
		{
			return;
		}

		$paths = new PathCollection(
			array(
				WINDWALKER . '/bundles',
				WINDWALKER_BUNDLE,
			)
		);

		$bundles = $paths->find('Bundle$');

		$config = $container->get('windwalker.config');

		foreach ($bundles as $bundle)
		{
			$bundleName = $bundle->getBasename();

			$class = $bundleName . '\\' . $bundleName;

			\JLoader::registerNamespace($bundleName, dirname((string) $bundle));

			if (class_exists($class) && is_subclass_of($class, 'Windwalker\\Bundle\\AbstractBundle'))
			{
				$config->set('bundle.' . $bundleName, $class);

				$class::registerProvider($container);
			}
		}
	}

	/**
	 * Set autoload.
	 *
	 * @return  void
	 */
	public function autoload()
	{
		static $loaded = false;

		if ($loaded)
		{
			return;
		}

		// Load Composer
		include_once dirname(__DIR__) . '/vendor/autoload.php';

		// Load Windwalker framework
		\JLoader::registerNamespace('Windwalker', dirname(__DIR__));

		// Load some file out of nameing standard
		jimport('joomla.filesystem.file');
		jimport('joomla.filesystem.folder');
		jimport('joomla.filesystem.path');

		// Aliases
		class_alias('Windwalker\Router\CmsRoute', 'Windwalker\Router\Route');

		$loaded = true;
	}
}
